<?php
require_once __DIR__ . '/functions.php';

// Ensure user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}
$user = currentUser($pdo);

$cycleId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$cycle = getCycleById($pdo, $cycleId);
if (!$cycle) {
    redirect('dashboard.php');
}
$class = getClassById($pdo, $cycle['class_id']);
// Ensure the user has permission to access this class (teacher/coordinator/inspector/principal)
if (!$class || !userCanAccessClass($pdo, $cycle['class_id'], $user['id'])) {
    redirect('dashboard.php');
}

$lessons = getLessons($pdo, $cycleId);

// Compute average evaluation scores per lesson for charting
$lessonLabels = [];
$avgMotricites = [];
$avgIntelligibilites = [];
$avgSensibilites = [];
$avgSociabilites = [];
foreach ($lessons as $lsn) {
    $lessonLabels[] = $lsn['title'];
    $evals = getEvaluations($pdo, $lsn['id']);
    $sumMot = $sumInt = $sumSens = $sumSoc = 0;
    $countEval = 0;
    foreach ($evals as $ev) {
        // Accumulate if values exist
        if (is_numeric($ev['motricite'])) { $sumMot += $ev['motricite']; }
        if (is_numeric($ev['intelligibilite'])) { $sumInt += $ev['intelligibilite']; }
        if (is_numeric($ev['sensibilite'])) { $sumSens += $ev['sensibilite']; }
        if (is_numeric($ev['sociabilite'])) { $sumSoc += $ev['sociabilite']; }
        $countEval++;
    }
    if ($countEval > 0) {
        $avgMotricites[] = $sumMot / $countEval;
        $avgIntelligibilites[] = $sumInt / $countEval;
        $avgSensibilites[] = $sumSens / $countEval;
        $avgSociabilites[] = $sumSoc / $countEval;
    } else {
        $avgMotricites[] = 0;
        $avgIntelligibilites[] = 0;
        $avgSensibilites[] = 0;
        $avgSociabilites[] = 0;
    }
}

$pageTitle = 'Cycle: ' . $cycle['name'];
include 'header.php';
?>

<div class="mt-4">
<h2>Détails du cycle : <?php echo htmlspecialchars($cycle['name']); ?></h2>

<p><strong>Classe :</strong> <?php echo htmlspecialchars($class['name']); ?></p>
<p><strong>Activité support (PPSAD) :</strong> <?php echo htmlspecialchars($cycle['ppsad']); ?></p>
<p><strong>Objet d'enseignement :</strong><br><?php echo nl2br(htmlspecialchars($cycle['objective'])); ?></p>
<?php if (!empty($cycle['oe_id'])): ?>
    <?php $oeInfo = getOEById($pdo, $cycle['oe_id']); ?>
    <?php if ($oeInfo): ?>
        <p><strong>Obj. de référence (bibliothèque) :</strong> <?php echo htmlspecialchars($oeInfo['ppsad'] . ' – ' . $oeInfo['name']); ?></p>
    <?php endif; ?>
<?php endif; ?>
<p><strong>Type d'expérience (ECC) :</strong> <?php echo htmlspecialchars($cycle['ecc_type']); ?></p>
<?php if ($cycle['start_date'] || $cycle['end_date']): ?>
    <p><strong>Période :</strong> <?php echo htmlspecialchars($cycle['start_date'] . ' - ' . $cycle['end_date']); ?></p>
<?php endif; ?>

<h3 class="mt-4">Leçons</h3>
<?php if (empty($lessons)): ?>
    <p>Aucune leçon n'est créée pour ce cycle. <a href="create_lesson.php?cycle_id=<?php echo $cycleId; ?>">Ajouter une leçon</a>.</p>
<?php else: ?>
    <div class="table-responsive">
        <table class="table table-sm table-striped align-middle datatable">
            <thead class="table-light">
                <tr>
                    <th>Date</th>
                    <th>Titre</th>
                    <th>FPS (forme de pratique scolaire)</th>
                    <th>CP (contenus prioritaires)</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ($lessons as $lesson): ?>
                <tr>
                    <td><?php echo htmlspecialchars($lesson['lesson_date']); ?></td>
                    <td><?php echo htmlspecialchars($lesson['title']); ?></td>
                    <td><?php echo nl2br(htmlspecialchars($lesson['fps'])); ?></td>
                    <td><?php echo nl2br(htmlspecialchars($lesson['cp'])); ?></td>
                    <td>
                        <a href="lesson_detail.php?id=<?php echo $lesson['id']; ?>" class="btn btn-sm btn-outline-primary">Voir / évaluer</a>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    <p><a href="create_lesson.php?cycle_id=<?php echo $cycleId; ?>">Ajouter une leçon</a></p>
    <p>
        <a href="export_cycle.php?cycle_id=<?php echo $cycleId; ?>">Exporter les évaluations du cycle (CSV)</a>
    </p>

    <!-- Chart showing the evolution of average evaluation scores across lessons -->
    <h4 class="mt-4">Progression des évaluations du cycle</h4>
    <div class="my-3">
        <canvas id="cycleEvaluationChart" height="200"></canvas>
    </div>
    <script>
    (function() {
        const ctx = document.getElementById('cycleEvaluationChart').getContext('2d');
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($lessonLabels, JSON_UNESCAPED_UNICODE); ?>,
                datasets: [
                    {
                        label: 'Motricité',
                        data: <?php echo json_encode($avgMotricites); ?>,
                        borderColor: 'rgba(255, 99, 132, 1)',
                        backgroundColor: 'rgba(255, 99, 132, 0.2)',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: 'Intelligibilité',
                        data: <?php echo json_encode($avgIntelligibilites); ?>,
                        borderColor: 'rgba(54, 162, 235, 1)',
                        backgroundColor: 'rgba(54, 162, 235, 0.2)',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: 'Sensibilité',
                        data: <?php echo json_encode($avgSensibilites); ?>,
                        borderColor: 'rgba(255, 206, 86, 1)',
                        backgroundColor: 'rgba(255, 206, 86, 0.2)',
                        tension: 0.3,
                        fill: false
                    },
                    {
                        label: 'Sociabilité',
                        data: <?php echo json_encode($avgSociabilites); ?>,
                        borderColor: 'rgba(75, 192, 192, 1)',
                        backgroundColor: 'rgba(75, 192, 192, 0.2)',
                        tension: 0.3,
                        fill: false
                    }
                ]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 20,
                        title: {
                            display: true,
                            text: 'Note moyenne (0-20)'
                        }
                    }
                },
                plugins: {
                    legend: {
                        position: 'top'
                    }
                }
            }
        });
    })();
    </script>
<?php endif; ?>

</div>

<?php
include 'footer.php';
?>