<?php
require_once __DIR__ . '/functions.php';

// Only logged in users can access
if (!isLoggedIn()) {
    redirect('login.php');
}
$user = currentUser($pdo);

// Restrict access to roles allowed to view reports: teacher, coordinator, principal, admin
if (!userHasRole($pdo, 'teacher') && !userHasRole($pdo, 'coordinator') && !userHasRole($pdo, 'principal') && !userHasRole($pdo, 'admin')) {
    redirect('dashboard.php');
}

// Compute ECC coverage per class
$classSummaries = [];
$totalCoverage = [];

// Get classes for this teacher (or all if admin)
if (userHasRole($pdo, 'admin')) {
    // Admin: fetch all classes
    $stmt = $pdo->query('SELECT * FROM classes');
    $classes = $stmt->fetchAll(PDO::FETCH_ASSOC);
} else {
    // Teachers and coordinators see their classes via mapping
    $classes = getTeacherClasses($pdo, $user['id']);
}

foreach ($classes as $cls) {
    $cycles = getCycles($pdo, $cls['id']);
    $eccCounts = [];
    foreach ($cycles as $cyc) {
        $ecc = trim($cyc['ecc_type']);
        if ($ecc === '') {
            $ecc = '(Non renseigné)';
        }
        if (!isset($eccCounts[$ecc])) {
            $eccCounts[$ecc] = 0;
        }
        $eccCounts[$ecc]++;
        // accumulate total
        if (!isset($totalCoverage[$ecc])) {
            $totalCoverage[$ecc] = 0;
        }
        $totalCoverage[$ecc]++;
    }
    $classSummaries[] = [
        'class' => $cls['name'],
        'ecc_counts' => $eccCounts,
        'total_cycles' => count($cycles)
    ];
}

// Prepare data for global ECC distribution chart
$eccLabels = array_keys($totalCoverage);
$eccCounts = array_values($totalCoverage);

$pageTitle = 'Rapports d\'analyse';
include 'header.php';
?>

<h2>Rapports et analyses</h2>

<p>Ce tableau indique la répartition des cycles par type d'expérience corporelle et culturelle (ECC) pour chacune de vos classes. Les ECC reflètent les grandes familles d'expériences (confrontation, dépassement, arts, aventure, acrobatie, soin de soi, etc.).
Lorsque l'ECC n'est pas renseignée sur un cycle, il apparaît sous la rubrique « Non renseigné ».</p>

<?php if (empty($classSummaries)): ?>
    <p>Aucune classe ou aucun cycle n'a été trouvé.</p>
<?php else: ?>
    <?php foreach ($classSummaries as $summary): ?>
        <h3>Classe : <?php echo htmlspecialchars($summary['class']); ?> (<?php echo $summary['total_cycles']; ?> cycle(s))</h3>
        <?php if (empty($summary['ecc_counts'])): ?>
            <p>Aucun cycle n'a été saisi pour cette classe.</p>
        <?php else: ?>
            <ul>
            <?php foreach ($summary['ecc_counts'] as $ecc => $count): ?>
                <li><strong><?php echo htmlspecialchars($ecc); ?> :</strong> <?php echo $count; ?> cycle(s)</li>
            <?php endforeach; ?>
            </ul>
        <?php endif; ?>
    <?php endforeach; ?>

    <h3>Récapitulatif global</h3>
    <p>Nombre total de cycles saisis : <strong><?php echo array_sum($totalCoverage); ?></strong></p>
    <ul>
    <?php foreach ($totalCoverage as $ecc => $count): ?>
        <li><strong><?php echo htmlspecialchars($ecc); ?> :</strong> <?php echo $count; ?> cycle(s)</li>
    <?php endforeach; ?>
    </ul>

    <!-- Chart for global ECC distribution -->
    <h4 class="mt-4">Répartition globale des cycles par ECC</h4>
    <div class="my-3">
        <canvas id="eccChart" height="150"></canvas>
    </div>
    <script>
    (function() {
        const ctx = document.getElementById('eccChart').getContext('2d');
        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($eccLabels, JSON_UNESCAPED_UNICODE); ?>,
                datasets: [{
                    label: 'Nombre de cycles',
                    data: <?php echo json_encode($eccCounts); ?>,
                    backgroundColor: 'rgba(75, 192, 192, 0.5)',
                    borderColor: 'rgba(75, 192, 192, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Nombre de cycles'
                        }
                    }
                },
                plugins: {
                    legend: { display: false }
                }
            }
        });
    })();
    </script>
<?php endif; ?>

<?php
include 'footer.php';
?>