<?php
/**
 * Common functions used across the application
 */

require_once __DIR__ . '/config.php';

session_start();

/**
 * Redirect to a given URL and exit
 *
 * @param string $url
 */
function redirect($url)
{
    header('Location: ' . $url);
    exit;
}

/**
 * Check if a user is logged in
 *
 * @return bool
 */
function isLoggedIn()
{
    return isset($_SESSION['user_id']);
}

/**
 * Retrieve the currently logged in user
 *
 * @param PDO $pdo
 * @return array|null
 */
function currentUser(PDO $pdo)
{
    if (!isLoggedIn()) {
        return null;
    }
    $stmt = $pdo->prepare('SELECT * FROM users WHERE id = ?');
    $stmt->execute([$_SESSION['user_id']]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Log in a user with email and password
 *
 * @param PDO    $pdo
 * @param string $email
 * @param string $password
 * @return bool true on success, false otherwise
 */
function loginUser(PDO $pdo, $email, $password)
{
    $stmt = $pdo->prepare('SELECT * FROM users WHERE email = ?');
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['user_id'] = $user['id'];
        return true;
    }
    return false;
}

/**
 * Register a new user (teacher by default)
 *
 * @param PDO    $pdo
 * @param string $name
 * @param string $email
 * @param string $password
 * @param string $role
 * @return bool true on success, false otherwise
 */
function registerUser(PDO $pdo, $name, $email, $password, $role = 'teacher')
{
    // Check if email already exists
    $stmt = $pdo->prepare('SELECT id FROM users WHERE email = ?');
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        return false;
    }
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)');
    $result = $stmt->execute([$name, $email, $hash, $role]);
    if ($result) {
        $userId = $pdo->lastInsertId();
        // If the user_roles table exists, assign the selected role to the user
        $tables = $pdo->query("SHOW TABLES LIKE 'user_roles'")->fetchAll(PDO::FETCH_ASSOC);
        if ($tables) {
            // Get role id from roles table
            $stmtRole = $pdo->prepare('SELECT id FROM roles WHERE name = ? LIMIT 1');
            $stmtRole->execute([$role]);
            $roleId = $stmtRole->fetchColumn();
            if ($roleId) {
                $stmtMap = $pdo->prepare('INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)');
                $stmtMap->execute([$userId, $roleId]);
            }
        }
    }
    return $result;
}

/**
 * Log out the current user
 */
function logoutUser()
{
    session_unset();
    session_destroy();
}

/**
 * Get all classes for a teacher
 *
 * @param PDO $pdo
 * @param int $teacherId
 * @return array
 */
function getClasses(PDO $pdo, $teacherId)
{
    $stmt = $pdo->prepare('SELECT * FROM classes WHERE teacher_id = ? ORDER BY created_at DESC');
    $stmt->execute([$teacherId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get class details
 *
 * @param PDO $pdo
 * @param int $classId
 * @return array|null
 */
function getClassById(PDO $pdo, $classId)
{
    $stmt = $pdo->prepare('SELECT * FROM classes WHERE id = ?');
    $stmt->execute([$classId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get cycles for a class
 *
 * @param PDO $pdo
 * @param int $classId
 * @return array
 */
function getCycles(PDO $pdo, $classId)
{
    $stmt = $pdo->prepare('SELECT * FROM cycles WHERE class_id = ? ORDER BY created_at DESC');
    $stmt->execute([$classId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get a cycle by ID
 *
 * @param PDO $pdo
 * @param int $cycleId
 * @return array|null
 */
function getCycleById(PDO $pdo, $cycleId)
{
    $stmt = $pdo->prepare('SELECT * FROM cycles WHERE id = ?');
    $stmt->execute([$cycleId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get lessons for a cycle
 *
 * @param PDO $pdo
 * @param int $cycleId
 * @return array
 */
function getLessons(PDO $pdo, $cycleId)
{
    $stmt = $pdo->prepare('SELECT * FROM lessons WHERE cycle_id = ? ORDER BY lesson_date ASC');
    $stmt->execute([$cycleId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get lesson by ID
 *
 * @param PDO $pdo
 * @param int $lessonId
 * @return array|null
 */
function getLessonById(PDO $pdo, $lessonId)
{
    $stmt = $pdo->prepare('SELECT * FROM lessons WHERE id = ?');
    $stmt->execute([$lessonId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get students of a class
 *
 * @param PDO $pdo
 * @param int $classId
 * @return array
 */
function getStudents(PDO $pdo, $classId)
{
    $stmt = $pdo->prepare('SELECT * FROM students WHERE class_id = ? ORDER BY name ASC');
    $stmt->execute([$classId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get evaluations for a lesson
 *
 * @param PDO $pdo
 * @param int $lessonId
 * @return array
 */
function getEvaluations(PDO $pdo, $lessonId)
{
    $stmt = $pdo->prepare(
        'SELECT e.*, s.name AS student_name FROM evaluations e
         JOIN students s ON e.student_id = s.id
         WHERE e.lesson_id = ? ORDER BY s.name ASC'
    );
    $stmt->execute([$lessonId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get all roles (social roles) defined in the system.
 *
 * @param PDO $pdo
 * @return array
 */
function getRoles(PDO $pdo)
{
    $stmt = $pdo->query('SELECT * FROM roles ORDER BY name ASC');
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get role assignments for a lesson.
 *
 * @param PDO $pdo
 * @param int $lessonId
 * @return array keyed by student_id
 */
function getLessonRoles(PDO $pdo, $lessonId)
{
    $stmt = $pdo->prepare(
        'SELECT lr.id, lr.student_id, lr.role_id, r.name AS role_name FROM lesson_roles lr
         JOIN roles r ON lr.role_id = r.id
         WHERE lr.lesson_id = ?'
    );
    $stmt->execute([$lessonId]);
    $roles = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $roles[$row['student_id']] = $row;
    }
    return $roles;
}

/**
 * Determine whether a user has access to a given class. Teachers have access to
 * classes mapped in class_teachers or via the legacy teacher_id field. Coordinators,
 * principals and inspectors have read access to all classes. Students do not have access via this function.
 *
 * @param PDO $pdo
 * @param int $classId
 * @param int $userId
 * @return bool
 */
function userCanAccessClass(PDO $pdo, $classId, $userId)
{
    // Check high privilege roles
    $roles = getUserRoles($pdo, $userId);
    if (in_array('coordinator', $roles) || in_array('principal', $roles) || in_array('inspector', $roles) || in_array('admin', $roles)) {
        return true;
    }
    // Check mapping table
    $tables = $pdo->query("SHOW TABLES LIKE 'class_teachers'")->fetchAll(PDO::FETCH_ASSOC);
    if ($tables) {
        $stmt = $pdo->prepare('SELECT 1 FROM class_teachers WHERE class_id = ? AND user_id = ?');
        $stmt->execute([$classId, $userId]);
        if ($stmt->fetch()) {
            return true;
        }
    }
    // Fallback to legacy teacher_id field
    $stmt = $pdo->prepare('SELECT 1 FROM classes WHERE id = ? AND teacher_id = ?');
    $stmt->execute([$classId, $userId]);
    return (bool)$stmt->fetch();
}

/**
 * Retrieve all role names assigned to a given user via the user_roles mapping.
 *
 * @param PDO $pdo
 * @param int $userId
 * @return array list of role names (e.g. 'teacher', 'student', 'coordinator', 'inspector', 'principal', 'admin')
 */
function getUserRoles(PDO $pdo, $userId)
{
    // If the legacy users table still has a single role column, include that role as well
    $roles = [];
    // Pull roles from the user_roles join table if it exists
    $tables = $pdo->query("SHOW TABLES LIKE 'user_roles'")->fetchAll(PDO::FETCH_ASSOC);
    if ($tables) {
        $stmt = $pdo->prepare('SELECT r.name FROM user_roles ur JOIN roles r ON ur.role_id = r.id WHERE ur.user_id = ?');
        $stmt->execute([$userId]);
        $roles = $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    // Fallback to role column in users table for backward compatibility
    $stmt = $pdo->prepare('SELECT role FROM users WHERE id = ?');
    $stmt->execute([$userId]);
    $legacyRole = $stmt->fetchColumn();
    if ($legacyRole && !in_array($legacyRole, $roles)) {
        $roles[] = $legacyRole;
    }
    return $roles;
}

/**
 * Check if the currently logged in user has a specific role
 *
 * @param PDO    $pdo
 * @param string $roleName
 * @return bool
 */
function userHasRole(PDO $pdo, $roleName)
{
    if (!isLoggedIn()) {
        return false;
    }
    $roles = getUserRoles($pdo, $_SESSION['user_id']);
    return in_array($roleName, $roles);
}

/**
 * Retrieve classes accessible to a teacher based on the class_teachers mapping.
 * Falls back to the legacy teacher_id column in classes if class_teachers does not exist.
 *
 * @param PDO $pdo
 * @param int $userId
 * @return array list of classes
 */
function getTeacherClasses(PDO $pdo, $userId)
{
    // If class_teachers table exists, use it for many-to-many relationships
    $tables = $pdo->query("SHOW TABLES LIKE 'class_teachers'")->fetchAll(PDO::FETCH_ASSOC);
    if ($tables) {
        $stmt = $pdo->prepare('SELECT c.* FROM classes c JOIN class_teachers ct ON c.id = ct.class_id WHERE ct.user_id = ? ORDER BY c.created_at DESC');
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    // Fallback: legacy one-to-one teacher_id field
    return getClasses($pdo, $userId);
}

/**
 * Assign roles to students for a lesson.  This function will
 * overwrite existing assignments.
 *
 * @param PDO   $pdo
 * @param int   $lessonId
 * @param array $assignments associative array student_id => role_id
 */
function setLessonRoles(PDO $pdo, $lessonId, array $assignments)
{
    // Remove previous assignments
    $stmt = $pdo->prepare('DELETE FROM lesson_roles WHERE lesson_id = ?');
    $stmt->execute([$lessonId]);
    // Insert new assignments
    $stmtIns = $pdo->prepare('INSERT INTO lesson_roles (lesson_id, student_id, role_id) VALUES (?, ?, ?)');
    foreach ($assignments as $studentId => $roleId) {
        if ($roleId) {
            $stmtIns->execute([$lessonId, $studentId, $roleId]);
        }
    }
}

/**
 * Get all objects d'enseignement from the library.
 *
 * @param PDO $pdo
 * @return array
 */
function getOELibrary(PDO $pdo)
{
    $stmt = $pdo->query('SELECT * FROM oe_library ORDER BY ppsad, name');
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get a single OE from the library by ID.
 *
 * @param PDO $pdo
 * @param int $oeId
 * @return array|null
 */
function getOEById(PDO $pdo, $oeId)
{
    $stmt = $pdo->prepare('SELECT * FROM oe_library WHERE id = ?');
    $stmt->execute([$oeId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get all forms of practice from the FPS library.
 *
 * @param PDO $pdo
 * @return array
 */
function getFPSLibrary(PDO $pdo)
{
    $stmt = $pdo->query('SELECT * FROM fps_library ORDER BY ppsad, name');
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get a single FPS entry from the library by ID.
 *
 * @param PDO $pdo
 * @param int $fpsId
 * @return array|null
 */
function getFPSById(PDO $pdo, $fpsId)
{
    $stmt = $pdo->prepare('SELECT * FROM fps_library WHERE id = ?');
    $stmt->execute([$fpsId]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
