<?php
/**
 * Page d'importation des élèves depuis un fichier CSV
 *
 * Cette page permet à un enseignant d'importer rapidement plusieurs élèves
 * pour une classe à partir d'un fichier CSV. Le fichier doit comporter
 * au minimum un nom et un prénom séparés par des points‑virgules. Des
 * colonnes supplémentaires peuvent inclure la date de naissance, le sexe
 * et des commentaires. Ces informations seront stockées dans la table
 * `students` si les colonnes existent dans le schéma.
 */

require_once __DIR__ . '/functions.php';

// Vérifier que l'utilisateur est connecté
if (!isLoggedIn()) {
    redirect('login.php');
}

// Récupérer l'utilisateur courant et la classe
$user = currentUser($pdo);
$classId = isset($_GET['class_id']) ? (int)$_GET['class_id'] : 0;
$class = getClassById($pdo, $classId);

// Vérifier l'existence de la classe et la propriété
if (!$class || $class['teacher_id'] !== $user['id']) {
    redirect('dashboard.php');
}

$message = '';
$error = '';

// Traitement du formulaire d'import
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Vérifier la présence d'un fichier CSV
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
        $fileTmpPath = $_FILES['csv_file']['tmp_name'];
        $filename = $_FILES['csv_file']['name'];
        $fileExt = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if ($fileExt !== 'csv') {
            $error = "Le fichier doit être au format CSV.";
        } else {
            // Ouvrir le fichier CSV et parcourir les lignes
            if (($handle = fopen($fileTmpPath, 'r')) !== false) {
                $count = 0;
                $rowIndex = 0;
                while (($data = fgetcsv($handle, 1000, ';')) !== false) {
                    $rowIndex++;
                    // Ignorer la ligne d'entête si elle contient 'Nom' ou 'nom'
                    if ($rowIndex === 1 && isset($data[0])) {
                        $firstCell = trim(mb_strtolower($data[0], 'UTF-8'));
                        if ($firstCell === 'nom' || $firstCell === 'name') {
                            continue;
                        }
                    }
                    // Récupérer les champs (nom, prénom, date de naissance, sexe, commentaires)
                    $nom = isset($data[0]) ? trim($data[0]) : '';
                    $prenom = isset($data[1]) ? trim($data[1]) : '';
                    $dob = isset($data[2]) ? trim($data[2]) : null;
                    $gender = isset($data[3]) ? trim($data[3]) : null;
                    $comments = isset($data[4]) ? trim($data[4]) : null;
                    // Construire le nom complet
                    $fullName = trim($nom . ' ' . $prenom);
                    if ($fullName === '') {
                        // Ignorer les lignes sans nom
                        continue;
                    }
                    // Insérer l'élève dans la base.  On utilise les colonnes si elles existent.
                    $stmt = $pdo->prepare('INSERT INTO students (class_id, name, date_of_birth, gender, comments) VALUES (?, ?, ?, ?, ?)');
                    $stmt->execute([
                        $classId,
                        $fullName,
                        $dob !== '' ? $dob : null,
                        $gender !== '' ? $gender : null,
                        $comments !== '' ? $comments : null,
                    ]);
                    $count++;
                }
                fclose($handle);
                $message = $count . ' élève' . ($count > 1 ? 's' : '') . ' importé' . ($count > 1 ? 's' : '') . ' avec succès';
            } else {
                $error = "Impossible de lire le fichier CSV.";
            }
        }
    } else {
        $error = "Veuillez sélectionner un fichier CSV à importer.";
    }
}

// Préparer le titre de la page
$pageTitle = 'Importer des élèves';
include 'header.php';
?>

<h2>Importer des élèves</h2>

<p>Classe : <strong><?php echo htmlspecialchars($class['name']); ?></strong></p>

<?php if ($message): ?>
    <div class="message success"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>
<?php if ($error): ?>
    <div class="message error"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<p>Le fichier CSV doit contenir les colonnes suivantes : <code>Nom;Prénom;Date de naissance;Sexe;Commentaires</code>. Seules les deux premières colonnes sont obligatoires. La date doit être au format AAAA‑MM‑JJ pour être correctement enregistrée.</p>

<form method="post" enctype="multipart/form-data" action="">
    <label for="csv_file">Sélectionner le fichier CSV</label><br>
    <input type="file" name="csv_file" id="csv_file" accept=".csv" required><br><br>
    <button type="submit">Importer</button>
</form>

<p><a href="class_detail.php?id=<?php echo $classId; ?>">← Retour à la classe</a></p>

<?php include 'footer.php'; ?>